/* $Id: testmem.c,v 1.14 1998/07/20 21:13:04 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Chris Sutton and Don Hiller */
                     
#include "sema.h"

/*
 *************************************************************************
 test_32_data_lines()
 
 Does a read & write of a marching 1 and zero pattern to test
 data lines.   Set the page map reg before calling this.
   
   > mn		module list node - module being tested
   > dspAddr  	96000 DSP address.  MS bits ignored.   
   > busName  	net name for error message
   > printErrs  print errors
   < returns    0 if ok, -1 if berr, >0 count of errors   
 
 *************************************************************************
 */
 
static SHORTSIZ16
test_32_data_lines(E1432_MODULE_LIST_NODE *mn, ULONGSIZ32 dspAddr,
  char *busName, int printErrs )
{                             
    SHORTSIZ16 err = 0;
    SHORTSIZ16 erCount = 0;
    SHORTSIZ16 bit;       
    ULONGSIZ32 testAddr;   /* a24 VXI addres offset from mod A24 */
    ULONGSIZ32 otherAddr;  /* a24 VXI addres offset from mod A24 */
    ULONGSIZ32 writeVal, readVal;
    LONGSIZ32 badBits = 0; /* one bit set for each bad data line */
    int numBadBits = 0;        /* number of bits set in badBits */

    DIAG_PRINTF(2, ("Testing %s data lines.\n", busName));
     /* convert 96000 DSP address to VXI A24 module address in 
     * the "movable DSP bus window" 
     */                                             
    testAddr = dspAddr << 2L;
    if (mn->a24_256k)
    {
	testAddr &= 0x1ffffUL;
	/* testAddr += 0x20000UL;  hostlib doesn't map the base window */
    }
    else
    {
	testAddr &= 0x7ffffUL;
	testAddr += 0x80000UL;
    }
    otherAddr = testAddr + 0xf0;     
    for (bit = 0; bit <32; bit++) {     
        writeVal = (1L << bit);
        i1432_w32(mn, &err, testAddr, writeVal);
        i1432_w32(mn, &err, otherAddr, ~writeVal);
        i1432_r32u(mn, &err, testAddr, &readVal);
	if ( err ) return err;
    	if (writeVal != readVal) {
    	    erCount++;
	    /* save EX-OR of expected and got */
    	    badBits |= (writeVal ^ readVal);
    	    if (erCount == 1 && printErrs)  {	/* first error */
                DIAG_PRINTF(1, (" Key to data bits:  0x------**"
		  "  %s(7) ..%s(0) \n", busName, busName));
                DIAG_PRINTF(1, ("                    0x----**--"
		  "  %s(15)..%s(8) \n", busName, busName));
                DIAG_PRINTF(1, ("                    0x--**----"
		  "  %s(23)..%s(16)\n", busName, busName));
                DIAG_PRINTF(1, ("                    0x**------"
		  "  %s(31)..%s(24)\n", busName, busName));
    	    }               
    	    if (erCount < 9 && printErrs)  {
                DIAG_PRINTF(1, ("          expected: 0x%08lX,\n", writeVal));
                DIAG_PRINTF(1, ("               got: 0x%08lX.\n", readVal));
	    }    	
	}
        /* if (diag_stop_testing()) return DIAG_ABORT; */
    }
    if (erCount>=9 && printErrs) {
	DIAG_PRINTF(1, ("...and more\n"));
    }                                                          
    
    /* count bad bits */
    numBadBits = 0;
    for (bit = 0; bit <32; bit++) 
    	if (badBits & (1L << bit))
    	    numBadBits++;
    
    if ((numBadBits > 0) && (numBadBits < 10) && printErrs) {
	/* print bad bits */
        DIAG_PRINTF(1, ("\n"));
     	DIAG_PRINTF(1, (" Bad lines:  "));
     	for (bit = 0; bit <32; bit++) {
	    if (badBits & (1L << bit))
	    	DIAG_PRINTF(1, ("%s(%d)  ", busName, bit ));     	     
     	}
     	DIAG_PRINTF(1, ("\n\n"))
    }
    return erCount;
}

/*
 *************************************************************************
 test_addr_lines()
 
  Tests memory by read and writes from addresses generated by
  marching a 1 through the address lines.   
  
   > mn			module list node - module being tested
   > dspStartAddr  	96000 DSP start address.  MS bits ignored.   
   > PageMapStart	Page map reg value for start address.
   > NumAddrBits  	Number of address bits to test.  Bits
   			greater than 17 (15 if 256k A24) are set using
			the Page Map register.
   > busName  	net name for error message
   > printErrs  print errors
   < returns    0 if ok, -1 if berr, >0 count of errors   
 
 *************************************************************************
 */
 
static SHORTSIZ16
test_addr_lines(E1432_MODULE_LIST_NODE *mn, ULONGSIZ32 dspStartAddr,
  SHORTSIZ16 pageMapStart, SHORTSIZ16 numAddrBits, char *busName, int printErrs )
{                             
     SHORTSIZ16 err = 0;
     SHORTSIZ16 erCount = 0;
     LONGSIZ32 bit;       
     ULONGSIZ32 vxiStartAddr;   /* a24 VXI address offset from mod A24 */
     ULONGSIZ32 vxiTestAddr;  
     ULONGSIZ32 dspTestAddr; 
     SHORTSIZ16 pageMapTest;
     ULONGSIZ32 writeVal, readVal;
     int topbit;
     ULONGSIZ32 window, window_mask;

    DIAG_PRINTF(2, ("Testing %s address lines.\n", busName));

    /* set page map register */
    err = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, pageMapStart);
    if ( err ) {
	i1432_print_acc_fail(1, err, "writing to PAGE_MAP register");
	DIAG_PRINTF(1, ("  Suspect VXI interface.\n"));
	return ERR1432_DIAG_ERR_SUBST;        
    }
                                                 
     /* convert 96000 DSP address to VXI A24 module address in 
      * the "movable DSP bus window" 
      */                                             
    vxiStartAddr = dspStartAddr << 2L;
    if (mn->a24_256k)
    {
	/* window = 0x20000UL; hostlib doesn't map the base window */
	window = 0x00000UL;
	window_mask = 0x20000UL - 1;
	topbit = 15;
    }
    else
    {
	window = 0x80000UL;
	window_mask = window - 1;
	topbit = 17;
    }
    vxiStartAddr &= window_mask;
    vxiStartAddr += window;

    /* write something unique to each address */
    for (bit = 0; bit < numAddrBits; bit++) {     

     	writeVal = (bit << 24) | (bit << 16);
     	writeVal |= (bit << 8) | (bit);
     	
	/* the 10.2 compiler does not compile this correctly...
     	dspTestAddr = dspStartAddr | (1L << bit);
	...but does OK on the following */
        dspTestAddr = dspStartAddr;
        dspTestAddr |= (1L << bit);
     	vxiTestAddr = window + ((dspTestAddr << 2L) & window_mask);
     	if (bit >= topbit)
        {
	    /* the 10.2 compiler does not compile this correctly...
     	    pageMapTest = (SHORTSIZ16)(pageMapStart | (1L << (bit-topbit)));
	    ...but does OK on the following */
            pageMapTest = (SHORTSIZ16)pageMapStart;
            pageMapTest |= (1L << (bit-topbit));
        }
     	else
     	    pageMapTest = pageMapStart;                                  

     	/* write to a location with one addr line set */
     	/* mess with the page map reg if needed */             
     	if (bit >= topbit) 
	{
            (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG,
	      pageMapTest);
        }
        DIAG_PRINTF(3, ("  Writing 0x%lx to 0x%lx (0x%lx).\n",
	  writeVal, vxiTestAddr, dspTestAddr));
        i1432_w32(mn, &err, vxiTestAddr, writeVal);

     	/* write to addr with no addr lines set */                                      
     	if (bit >= topbit) 
	{
            (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG,
	      pageMapStart);
	}
        DIAG_PRINTF(3, ("  Writing 0x%lx to 0x%lx.\n",
	  0xffffffffUL, vxiStartAddr));
        i1432_w32(mn, &err, vxiStartAddr, 0xffffffffUL);
    	if ( err ) return err;
	/* if (diag_stop_testing()) return DIAG_ABORT; */
    }
    
    /* restore page map regiseter */
    if (numAddrBits > topbit) 
    {
        (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG,
	  pageMapStart);
    }

    /* read and see if any value was overwritten */                                      
    for (bit = 0; bit < numAddrBits; bit++) {     
     	
     	writeVal  = (bit << 24) | (bit << 16);
     	writeVal |= (bit << 8) | (bit);     
     	
	/* the 10.2 compiler does not compile this correctly...
     	dspTestAddr = dspStartAddr | (1L << bit);
	...but does OK on the following */
        dspTestAddr = dspStartAddr;
        dspTestAddr |= (1L << bit);
     	vxiTestAddr = window + ((dspTestAddr << 2L) & window_mask);
     	if (bit >= topbit)
        {
	    /* the 10.2 compiler does not compile this correctly...
     	    pageMapTest = (SHORTSIZ16)(pageMapStart | (1L << (bit-topbit)));
	    ...but does OK on the following */
            pageMapTest = (SHORTSIZ16)pageMapStart;
            pageMapTest |= (1L << (bit-topbit));
        }
     	else
     	    pageMapTest = pageMapStart;                                  

     	/* mess with the page map reg if needed */             
     	if (bit >= topbit) 
	{
            (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG,
	      pageMapTest);
        }
     	
     	/* read it */
        i1432_r32u(mn, &err, vxiTestAddr, &readVal);
    	if ( err ) return err;
        if ( readVal != writeVal ) {
    	    erCount++;
            if ( printErrs ) {
                DIAG_PRINTF(1, ("   Expected: 0x%08lX,\n", writeVal));
                DIAG_PRINTF(1, ("        got: 0x%08lX at addr 0x%08lX.\n",
		  readVal, dspTestAddr));
	    }
        }
	else
	{
            DIAG_PRINTF(3, ("  Read 0x%lx at 0x%lx (0x%lx).\n",
	      readVal, vxiTestAddr, dspTestAddr));
	}
    	if (erCount>9 && printErrs) {
    	    DIAG_PRINTF(1, ("...maybe more\n"));
    	    break;      
    	}
        /* if (diag_stop_testing()) return DIAG_ABORT; */
     }

     if (erCount > 0 && printErrs) 	{
         DIAG_PRINTF(1, ("            Key to addr lines:  0x------**"
	   "  %s(7) ..%s(0) \n", busName, busName));
         DIAG_PRINTF(1, ("                                0x----**--"
	   "  %s(15)..%s(8) \n", busName, busName));
         DIAG_PRINTF(1, ("                                0x--**----"
	   "  %s(23)..%s(16)\n", busName, busName));
         DIAG_PRINTF(1, ("                                0x**------"
	   "  %s(31)..%s(24)\n", busName, busName));
         DIAG_PRINTF(1, ("\n"));
     }
     return erCount;
}


/*
 *************************************************************************
 e1432_test_bsram()
 
 Test the b-bus Static RAM.
 
 *************************************************************************
 */

SHORTSIZ16
i1432_test_bsram( E1432_MODULE_LIST_NODE *mn )
{
    SHORTSIZ16 err = 0;
    ULONGSIZ32 dspAddr;            
    SHORTSIZ16 numAddrBits;
    SHORTSIZ16 page;
    	/**************************
    	 * B SRAM data lines
    	 *************************/

    DIAG_PRINTF(1, ("Testing B-Bus Static RAM.\n"));

    page = E1432_B_SRAM_PAGE;
    if (mn->a24_256k)
	page <<= 2;

    /* set page map register to access B SRAM */    
    err = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page);
    
    dspAddr = 0x00020000UL;
    if ( err ) {
	i1432_print_acc_fail(1, err, "writing to PAGE_MAP register");
	DIAG_PRINTF(1, ("  Can't test B bus.  Suspect VXI interface.\n"));
	DIAG_PRINTF(0, ("  B SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    
    err = test_32_data_lines( mn, dspAddr, "bD", 1 );
    if ( err < 0 ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bus error while accessing B-bus \n"));
	DIAG_PRINTF(1, ("  Static RAM."
	  "  Failed on the first B-bus access.\n"));
	DIAG_PRINTF(1, ("  Suspect:"
	  "  B-bus decode EPLD U740, the 96000, signals \n"));
	DIAG_PRINTF(1, ("  like bVRn, VGn, bTAn. \n"));
	DIAG_PRINTF(0, ("  B SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    if (err) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bad data while testing the  \n"));
	DIAG_PRINTF(1, ("  B-bus Static RAM.  Suspect bD() lines.\n"));
	DIAG_PRINTF(1, ("  Check soldering of pins for FIT connector J1.\n"));
	DIAG_PRINTF(0, ("  B SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

    	/**************************
    	 * B SRAM address lines
    	 *************************/

    numAddrBits = 17;
    err = test_addr_lines( mn, dspAddr, page, numAddrBits, "bA", 1 );
    if ( err < 0 ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bus error while accessing B-bus \n"));
	DIAG_PRINTF(1, ("  Static RAM."
	  "  B-bus data line test worked, address test failed.\n"));
	DIAG_PRINTF(0, ("  B SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    if (err) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bad data while testing the  \n"));
	DIAG_PRINTF(1, ("  B-bus Static RAM.  Suspect bA() lines.\n"));
	DIAG_PRINTF(0, ("  B SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

    
    return 0;
}




/*
 *************************************************************************
 e1432_test_asram()
 
 Test the a-bus Static RAM.
 
 *************************************************************************
 */

SHORTSIZ16
i1432_test_asram( E1432_MODULE_LIST_NODE *mn )
{
    SHORTSIZ16 err = 0;
    ULONGSIZ32 dspAddr;            
    SHORTSIZ16 numAddrBits;
    SHORTSIZ16 page;

    	/**************************
    	 * a SRAM data lines
    	 *************************/

    DIAG_PRINTF(1, ("Testing A-Bus Static RAM.\n"));

    page = E1432_A_SRAM_PAGE;
    if (mn->a24_256k)
	page <<= 2;

    /* set page map register to access A SRAM */    
    err = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page);
    
    dspAddr = 0x00020000UL;
    if ( err ) {
	i1432_print_acc_fail(1, err, "writing to PAGE_MAP register");
	DIAG_PRINTF(1, ("  Can't test A bus.  Suspect VXI interface.\n"));
	DIAG_PRINTF(0, ("  A SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    
    err = test_32_data_lines( mn, dspAddr, "aD", 1 );
    if ( err < 0 ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bus error while accessing A-bus \n"));
	DIAG_PRINTF(1, ("  Static RAM.  Failed on the first A-bus access.\n"));
	DIAG_PRINTF(1, ("  Suspect:  DRAM EPLD U739, the 96000, signals \n"));
	DIAG_PRINTF(1, ("  like baVRn, VGn, aTSn. \n"));
	DIAG_PRINTF(0, ("  A SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    if (err) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bad data while testing the  \n"));
	DIAG_PRINTF(1, ("  A-bus Static RAM."
	  "  Suspect aD() lines, Bus Connect\n"));
	DIAG_PRINTF(1, ("  Buffers U618-626. \n"));
	DIAG_PRINTF(0, ("  A SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

    	/**************************
    	 * A SRAM address lines
    	 *************************/
    numAddrBits = 17;
    err = test_addr_lines( mn, dspAddr, page, numAddrBits, "aA", 1 );
    if ( err < 0 ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bus error while accessing A-bus \n"));
	DIAG_PRINTF(1, ("  Static RAM."
	  "  A-bus data line test worked, address test failed.\n"));
	DIAG_PRINTF(0, ("  A SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            
    if ( err ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bad data while testing the  \n"));
	DIAG_PRINTF(1, ("  A-bus Static RAM."
	  "  Suspect aA() lines,  Bus Connect\n"));
	DIAG_PRINTF(1, ("  Buffers U618-626. \n"));
	DIAG_PRINTF(0, ("  A SRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

    return 0;
}

/*
 *************************************************************************
 e1432_test_dram()
 
 Test the Dynamic RAM.  This test tests at least 4M DRAM, and more
 if more is found or DIAG_FLAGS_32M_DRAM is set.

  > *megaBytes	How many meg to test.  0, 4, 32.   -1 auto?
 
 *************************************************************************
 */

SHORTSIZ16
i1432_test_dram( E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 *megaBytes )
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 m;
    ULONGSIZ32 dspAddr;            
    SHORTSIZ16 numAddrBits;
    SHORTSIZ16 page;
    int printErrs = 1;

    if ( *megaBytes == E1432_DRAM_IGNORE ) return err;  /* nothing to test */

    /* hold off printing of errors while evaluating size */
    if ( *megaBytes == E1432_DRAM_UNKNOWN ) printErrs = 0;


    	/**************************
    	 * DRAM data lines on first page
    	 *************************/

    if ( *megaBytes == E1432_DRAM_UNKNOWN )
    {
        DIAG_PRINTF(1, ("Testing DRAM.\n"));
    }
    else
    {
        DIAG_PRINTF(1, ("Testing %d MB DRAM.\n", *megaBytes));
    }

    page = E1432_DRAM_PAGE;
    if (mn->a24_256k)
	page <<= 2;

    /* set page map register to access first page of DRAM */    
    err = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page);
    dspAddr = 0x01000000UL;
    if ( err ) {
	i1432_print_acc_fail(1, err, "writing to PAGE_MAP register");
	DIAG_PRINTF(1, ("  Can't test DRAM.  Suspect VXI interface.\n"));
	DIAG_PRINTF(0, ("  DRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }

    err = test_32_data_lines( mn, dspAddr, "aD", printErrs );
    if ( err < 0  && printErrs ) {
	i1432_print_acc_fail(1, err, "accessing DRAM");
	DIAG_PRINTF(1, ("  Failed on the first DRAM access.\n"));
	DIAG_PRINTF(1, ("  Suspect:  DRAM SIM, DRAM ADDER EPLD U755,"
	  " DRAM EPLD U739\n"));
	DIAG_PRINTF(0, ("  DRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_DRAM);        
    }
    if ( err ) {
	if ( *megaBytes == E1432_DRAM_UNKNOWN )
	{
	    /* no DRAM present, none expected, mark as none and return */
	    DIAG_PRINTF(1, ("  No DRAM found.\n"));
	    *megaBytes = E1432_DRAM_IGNORE;
	    return 0;
	}
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bad data while testing DRAM.\n"));
	DIAG_PRINTF(1, ("  Check:  DRAM SIM plugged in.\n"));
	DIAG_PRINTF(1, ("  Suspect aD() lines at SIM connector.\n"));
	DIAG_PRINTF(0, ("  DRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_DRAM);        
    }

    	/**************************
    	 * DRAM address lines 
    	 * on first page.
    	 *************************/
    /* if test_32_data_lines() passed, there are at least 4MB */

    numAddrBits = 17;                                 
    err = test_addr_lines( mn, dspAddr, page, numAddrBits, "DRAM_A", 1 );
    if ( err < 0 ) {
	i1432_print_acc_fail(1, err, "accessing DRAM");
	DIAG_PRINTF(1, ("  DRAM data line test worked,"
	  " address test failed.\n"));
	DIAG_PRINTF(0, ("  DRAM test failed.\n"));
	if ( err == ERR1432_DIAG_ERR_SUBST ) return DIAG_Q_ERR(err);
	else return DIAG_Q_ERR(ERR1432_DIAG_ERR_DRAM);        
    }                                                                            
    if ( err ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bad data while testing DRAM addressing.\n"));
	DIAG_PRINTF(1, ("  Suspect CAS_, RAS_, DRAM_A() lines,"
	  " DRAM ADDR EPLD U755.\n"));
	DIAG_PRINTF(1, ("  U755 routes aA(0:10)  to DRAM_A(0:10), \n"));
	DIAG_PRINTF(1, ("  and later   aA(11:18) to DRAM_A(0:10). \n"));
	DIAG_PRINTF(0, ("  DRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_DRAM);        
    }

    	/**************************
    	 * DRAM address lines 
    	 * on all pages.
    	 *************************/
    numAddrBits = 17;
    /* if unknown,  test for 32MB */
    if ( *megaBytes == E1432_DRAM_UNKNOWN ) m = E1432_DRAM_32M;
    else m = *megaBytes;
    while (m) {
        numAddrBits++;
	m >>= 1;
    }

    err = test_addr_lines(mn, dspAddr, page, numAddrBits, "DRAM_A", printErrs);
    if ( err < 0 && printErrs ) {
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bus error while accessing DRAM addressing.\n"));
	DIAG_PRINTF(1, ("  DRAM data line test passed,"
	  " address test passed on 1st page,\n"));
	DIAG_PRINTF(1, ("  failed on other pages.\n"));
	DIAG_PRINTF(0, ("  DRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_DRAM);        
    }
    if ( err ) {
	if ( *megaBytes == E1432_DRAM_UNKNOWN )
	{
	    /* 4MB DRAM present but not 32MB, mark as 4MB and return */
	    DIAG_PRINTF(1, ("  4MB DRAM found.\n"));
	    *megaBytes = E1432_DRAM_4M;
	    return 0;
	}
	DIAG_PRINTF(1, ("  Diagnostics failed."
	  "  Bad data while testing DRAM addressing.\n"));
	DIAG_PRINTF(1, ("  Test passed on 1st page, failed when testing"
	  " other pages.\n"));
	DIAG_PRINTF(0, ("  DRAM test failed.\n"));
	DIAG_PRINTF(1, ("\n"));
	if (*megaBytes > 4)
	  DIAG_PRINTF(1, ("  Check:  Correct DRAM SIMM."
	    "  Test expected %d Meg.\n", *megaBytes));
	DIAG_PRINTF(1, ("  Suspect CAS_, RAS_, DRAM_A() lines,"
	  " DRAM ADDR EPLD U755.\n"));
	DIAG_PRINTF(1, ("  DRAM_A lines are multiplexed:"
	  "  U755 first routes aA(0:10) \n"));
	DIAG_PRINTF(1, ("  to DRAM_A(0:10), and then aA(11:18)"
	  " to DRAM_A(0:10).\n"));
	DIAG_PRINTF(0, ("  DRAM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_DRAM);        
    }

    if ( *megaBytes == E1432_DRAM_UNKNOWN )
    {
	*megaBytes = E1432_DRAM_32M;
	DIAG_PRINTF(1, ("  %dMB DRAM found.\n", *megaBytes));
	return 0;
    }

    return 0;
}

/*
 *********************************************************************
 Test the revision value stored in the Flash ROM.
   > mn		module list node - module being tested
 *********************************************************************
 */
static SHORTSIZ16
test_boot_rev(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 err = 0;
    ULONGSIZ32 addr, tmp, rev;

    if (mn->a24_256k)
	/* addr = 0x020000UL + 0x108 * 4; hostlib doesn't map the base window */
	addr = 0x000000UL + 0x108 * 4;
    else
	addr = 0x080000UL + 0x40000UL + 0x108 * 4;

    i1432_r32u(mn, &err, addr, &tmp);
    rev = tmp & 0xff;
    i1432_r32u(mn, &err, addr + 4, &tmp);
    rev |= (tmp & 0xff) << 8;
    i1432_r32u(mn, &err, addr + 8, &tmp);
    rev |= (tmp & 0xff) << 16;
    i1432_r32u(mn, &err, addr + 12, &tmp);
    rev |= (tmp & 0xff) << 24;

    if ( err )
    {
	i1432_print_acc_fail(1, err, "reading Flash ROM revision");
	DIAG_PRINTF(1, ("  Earlier access to start of Flash ROM didn't"
	  " error.\n"));
	DIAG_PRINTF(1, ("  Can't get Flash ROM revision.\n"));
	DIAG_PRINTF(0, ("  FLASH PROM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    if (rev < 19950721 || rev > 20050000)
    {
	DIAG_PRINTF(1, ("  Expected Flash ROM rev >= 19950721, got %ld.\n"
	  , rev));
	DIAG_PRINTF(1, ("  Maybe ROM is OK but boot program needs to"
	  " be updated.\n"));
	DIAG_PRINTF(0, ("  FLASH PROM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    return 0;
}

/*
 *************************************************************************
 e1432_test_flash()
 
 Test the b-bus Flash ROM
 
 *************************************************************************
 */

#define READ_ID         0x90
#define READ_MODE       0xFF
#define FPROM_DEVICE_CODE 0x95

SHORTSIZ16
i1432_test_flash( E1432_MODULE_LIST_NODE *mn )
{
    SHORTSIZ16 err = 0;
    ULONGSIZ32 vxiAddr;   /* a24 VXI addres offset from mod A24 */
    ULONGSIZ32 readVal; 
    SHORTSIZ16 page;

    DIAG_PRINTF(1, ("Testing Flash ROM.\n"));

    page = E1432_FPROM_PAGE;
    if (mn->a24_256k)
    {
	page <<= 2;
	page += 2;
    }

    /* set page map register to access FROM */    
    err = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page);
    
    if ( err ) {
	i1432_print_acc_fail(1, err, "writing to PAGE_MAP register");
	DIAG_PRINTF(1, ("  Can't test Flash ROM.  Suspect VXI interface.\n"));
	DIAG_PRINTF(0, ("  FLASH PROM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            

    if (mn->a24_256k)
	/* vxiAddr = 0x020000UL; hostlib doesn't map the base window */
	vxiAddr = 0x000000UL;
    else
	vxiAddr = 0x080000UL + 0x40000UL;	/* boot area starts here */

    /* Write the  READ_ID command to the flash rom and expect it
     * to output the manufacturer id and device code 
     */
    i1432_w32(mn, &err, vxiAddr, READ_ID);

    /* read and discard Manufactuer ID so this will work with all vendors */
    i1432_r32u(mn, &err, vxiAddr, &readVal);

    /* read and check device ID */
    i1432_r32u(mn, &err, vxiAddr + 4, &readVal);

    i1432_w32(mn, &err, vxiAddr, READ_MODE);  /* switch ROM back to normal */

    if ( err ) {
	i1432_print_acc_fail(1, err, "accessing Flash ROM");
	DIAG_PRINTF(1, ("  Can't test Flash ROM.  Suspect VXI interface.\n"));
	DIAG_PRINTF(0, ("  FLASH PROM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                                                                            

    readVal &= 0xff;	/* flash rom is byte wide */
    if (readVal != FPROM_DEVICE_CODE) {
	DIAG_PRINTF(1, ("  Expected 0x%02X got 0x%02X.\n",
	  FPROM_DEVICE_CODE, (int) readVal));
	DIAG_PRINTF(1, ("  Diagnostics failed Flash ROM test. \n"));
	DIAG_PRINTF(1, ("  Bad data from the ROM when reading the ICs"
	  " DEVICE CODE.\n"));
	DIAG_PRINTF(1, ("  This should be correct even before the part is"
	  " programmed.\n"));
	DIAG_PRINTF(1, ("  Suspect the flash ROM U728.\n"));
	DIAG_PRINTF(0, ("  FLASH PROM test failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);        
    }                              

    err = test_boot_rev(mn);
    if (err) return err;

    return 0;
}
